Attribute VB_Name = "ErrorHandler"
' Error handler, developed by:
'
' Apex Consulting Group, Inc.
' 23 Walkers Brook Drive
' Suite 23
' Reading, MA 01867
'
' Tel: 617.489.9000
' Fax: 781.944.1988
'
' www.apexcgi.com

Option Explicit
Option Compare Text

Private Const MODULE_NAME     As String = "ErrorHandler"

Private mblnInitialized       As Boolean

'Log file information
Private mblnTracing           As Boolean
Private mstrLogFileFolder     As String
Private mstrLogFileNamePrefix As String
Private mintDaysToKeep        As Integer

'Error information
Private mlngErrNumber         As Long
Private mstrErrDescription    As String
Private mstrErrSource         As String
Private mstrDebugInfo         As String
Private mcolControlNameStack  As Collection

Private Declare Function GetPrivateProfileString Lib "kernel32" Alias "GetPrivateProfileStringA" (ByVal lpApplicationName As String, ByVal lpKeyName As Any, ByVal lpDefault As String, ByVal lpReturnedString As String, ByVal nSize As Long, ByVal lpFileName As String) As Long

Private Sub Initialize()
Const ERROR_HANDLER As String = "Postmortem"
Dim strIniFile      As String
    
    strIniFile = FixPath(App.Path) & ERROR_HANDLER & ".ini"
    
    mblnTracing = (GetIniString(ERROR_HANDLER, "Trace", strIniFile) = "True")
    mstrLogFileNamePrefix = GetIniString(ERROR_HANDLER, "FilenamePrefix", strIniFile)
    mstrLogFileFolder = FixPath(GetIniString(ERROR_HANDLER, "Folder", strIniFile))
    mintDaysToKeep = CInt(GetIniString(ERROR_HANDLER, "DaysToKeep", strIniFile))
    Set mcolControlNameStack = New Collection
    
    DeleteOldLogFiles
    
    mblnInitialized = True
    
End Sub

Private Function FixPath(ByVal vstrPath As String) As String
    
    If Len(vstrPath) > 0 Then
        If Right$(vstrPath, 1) = "\" Then
            FixPath = vstrPath
        Else
            FixPath = vstrPath & "\"
        End If
    End If
    
End Function

Private Function GetIniString(ByVal vstrSection As String, _
                             ByVal vstrKey As String, _
                             ByVal vstrIniFile As String) As String
Dim lngStringLength As Long
Dim strIniValue     As String * 255
    
    lngStringLength = GetPrivateProfileString(vstrSection, vstrKey, "", strIniValue, Len(strIniValue), vstrIniFile)

    If lngStringLength > 0 Then
        GetIniString = Left$(strIniValue, lngStringLength)
    End If

End Function

Private Sub DeleteOldLogFiles()
Dim objFileSystem      As Scripting.FileSystemObject
Dim strFileName        As String
Dim strPathAndFileName As String
Dim dtmCreateDate      As Date
Dim lngDays            As Long
    
    Set objFileSystem = New Scripting.FileSystemObject
    
    With objFileSystem
        strFileName = Dir(mstrLogFileFolder & mstrLogFileNamePrefix & "*.txt")
    
        While Len(strFileName) > 0
            strPathAndFileName = mstrLogFileFolder & strFileName
            dtmCreateDate = FileDateTime(strPathAndFileName)
            lngDays = DateDiff("d", dtmCreateDate, Now())
            If lngDays > mintDaysToKeep Then
                .DeleteFile strPathAndFileName
            End If
            strFileName = Dir
        Wend
    End With
    
    Set objFileSystem = Nothing

End Sub

Public Sub TrapError(ByVal vstrModule As String, _
                     ByVal vstrRoutine As String, _
                     ByVal vlngLineNumber As Long, _
                     Optional ByRef rstrDebugInfo As String)
                     
'Captures the error and builds the call stack without logging anything.
'Call this from nested routines.  Top level routines should use LogError.
    
    CaptureErrorInfo vstrModule, vstrRoutine, vlngLineNumber, rstrDebugInfo
    
    'Raise the error again so that it will bubble up through the call path.
    Err.Raise Number:=mlngErrNumber, _
              Description:=mstrErrDescription, _
              Source:=mstrErrSource
End Sub

Public Sub LogError(ByVal vstrModule As String, _
                    ByVal vstrRoutine As String, _
                    ByVal vlngLineNumber As Long, _
                    Optional ByRef rstrDebugInfo As String)
'Captures the error and writes it to the log file.
'Call this routine from the top level of any nested calls.
'Lower level routines should use the
'TrapError routine in order to capture the call stack.

    WriteLogError vstrModule, vstrRoutine, vlngLineNumber, rstrDebugInfo
    
    'Raise the error again so that the caller can detect it.
    Err.Raise Number:=mlngErrNumber, _
              Description:=mstrErrDescription, _
              Source:=mstrErrSource

End Sub

Private Sub WriteLogError(ByVal vstrModule As String, _
                          ByVal vstrRoutine As String, _
                          ByVal vlngLineNumber As Long, _
                          Optional ByRef rstrDebugInfo As String)
Dim strMsg As String
    
    CaptureErrorInfo vstrModule, vstrRoutine, vlngLineNumber, rstrDebugInfo
    
    strMsg = "Error in " & App.EXEName & vbCrLf & _
             "Date:" & vbTab & vbTab & Format$(Date, "mm/dd/yy") & vbCrLf & _
             "Time:" & vbTab & vbTab & Format$(Time, "hh:mm:ss") & vbCrLf & _
             "Error:" & vbTab & vbTab & CStr(mlngErrNumber) & vbCrLf & _
             "Description:" & vbTab & mstrErrDescription & vbCrLf & _
             "Module:" & vbTab & vbTab & vstrModule & vbCrLf & _
             "Routine:" & vbTab & vstrRoutine & vbCrLf & _
             "Source:" & vbTab & vbTab & mstrErrSource & vbCrLf
    
    If Len(mstrDebugInfo) > 0 Then
        strMsg = strMsg & "Debug information: " & vbCrLf & mstrDebugInfo
    End If
    
    mblnTracing = True
    
    WriteLog strMsg, False
    
    'Write the error to the NT event log as well in case the error was caused during
    'initialization, which could prevent writing to the application's own log.
    App.StartLogging "", vbLogToNT
    App.LogEvent "Error: " & CStr(mlngErrNumber) & " - " & mstrErrDescription, vbLogEventTypeError
    
End Sub

Private Sub CaptureErrorInfo(ByVal vstrModule As String, _
                             ByVal vstrRoutine As String, _
                             ByVal vlngLineNumber As Long, _
                             Optional ByRef rstrDebugInfo As String)
'Captures the error and builds the call stack.
Dim lngErrNumber      As Long
Dim strErrDescription As String
Dim strErrSource      As String
    
    
    'Save the error because it can be lost by executing other VB commands.
    With Err
        lngErrNumber = .Number
        strErrDescription = .Description
        strErrSource = .Source
        If strErrSource = App.EXEName Then
            strErrSource = ""
        End If
    End With
    
    'Capture only the original error in case it gets changed
    'as the error is raised back through the call path.
    'Build the calling path information by prepending the
    'module and routine names of each caller.
    If mlngErrNumber = 0 Then
        mlngErrNumber = lngErrNumber
        If Left$(strErrDescription, 7) = "(Line #" Then
            mstrErrDescription = strErrDescription
        Else
            mstrErrDescription = "(Line # " & CStr(vlngLineNumber) & ") " & strErrDescription
        End If
        mstrDebugInfo = rstrDebugInfo
    End If
    
    If Len(strErrSource) = 0 Then
        mstrErrSource = vstrModule & "." & vstrRoutine
    Else
        mstrErrSource = vstrModule & "." & vstrRoutine & " -> " & strErrSource
    End If
    
End Sub

Public Sub WriteLog(ByVal vstrMsg As String, _
                    Optional ByVal vblnTimestamp As Boolean = True)
'Appends a message to the log file.
Dim objFileSystem As Scripting.FileSystemObject
Dim objTextStream As Scripting.TextStream
Dim strLogFile    As String
   
    If Not mblnInitialized Then Initialize
   
    If Not mblnTracing Then Exit Sub
    
    strLogFile = mstrLogFileFolder & mstrLogFileNamePrefix & " " & Format$(Now, "yymmdd") & ".txt"

    Set objFileSystem = CreateObject("Scripting.FileSystemObject")
    Set objTextStream = objFileSystem.OpenTextFile(strLogFile, ForAppending, True, TristateFalse)
    
    If vblnTimestamp Then
        objTextStream.Write Format$(Now, "hh:mm:ss") & "  " & vstrMsg & vbCrLf
    Else
        objTextStream.Write vstrMsg & vbCrLf
    End If
    
    objTextStream.Close

End Sub

Public Sub StdGotFocus(robjActiveForm As Object)
'Standard routine called by all controls in their GotFocus event.
Dim objActiveControl As Control
Dim strIndex         As String
    
    On Error Resume Next
    Set objActiveControl = robjActiveForm.ActiveControl
    strIndex = "(" & objActiveControl.Index & ")"
    Err.Clear
    
    On Error GoTo Err
    
    'Capture the names of the last 10 controls that received the focus.
    '10 is an arbitrary number and can be changed if required.
    If Not objActiveControl Is Nothing Then
        With mcolControlNameStack
            .Add robjActiveForm.Name & "." & objActiveControl.Name & strIndex
            If .Count > 10 Then
                .Remove 1
            End If
        End With
        
'''        With objActiveControl
'''            Select Case True
'''                Case TypeOf objActiveControl Is TextBox
'''                    .SelStart = 0
'''                    .SelLength = Len(.Text)
'''                Case TypeOf objActiveControl Is MaskEdBox
'''                    .SelStart = 0
'''                    .SelLength = Len(.FormattedText)
'''            End Select
'''        End With
    End If
    
    Set objActiveControl = Nothing
    
    Exit Sub
    
Err:
    TrapError MODULE_NAME, "StdGotFocus", Erl
End Sub

